<?php

namespace App\Http\Controllers;

use App\Models\Ticket;
use App\Models\KnowledgeBase;
use App\Models\Setting;
use App\Models\TicketReply;
use App\Models\TicketAttachment;
use App\Mail\TicketReplyMail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;

class AdminController extends Controller
{
    public function dashboard()
    {
        $stats = [
            'total_tickets' => Ticket::count(),
            'open_tickets' => Ticket::where('status', 'open')->count(),
            'in_progress_tickets' => Ticket::where('status', 'in_progress')->count(),
            'closed_tickets' => Ticket::where('status', 'closed')->count(),
        ];

        $recentTickets = Ticket::with('user')->latest()->limit(10)->get();

        return view('admin.dashboard', compact('stats', 'recentTickets'));
    }

    public function settings()
    {
        $settings = [
            'logo' => Setting::get('logo'),
            'site_name' => Setting::get('site_name', 'Ticket System'),
            'meta_title' => Setting::get('meta_title'),
            'meta_description' => Setting::get('meta_description'),
            'timezone' => Setting::get('timezone', 'UTC'),
        ];

        return view('admin.settings', compact('settings'));
    }

    public function updateSettings(Request $request)
    {
        $validated = $request->validate([
            'site_name' => ['required', 'string', 'max:255'],
            'meta_title' => ['nullable', 'string', 'max:255'],
            'meta_description' => ['nullable', 'string'],
            'timezone' => ['required', 'string'],
            'logo' => ['nullable', 'image', 'max:2048'],
        ]);

        Setting::set('site_name', $validated['site_name']);
        Setting::set('meta_title', $validated['meta_title']);
        Setting::set('meta_description', $validated['meta_description']);
        Setting::set('timezone', $validated['timezone']);

        if ($request->hasFile('logo')) {
            $path = $request->file('logo')->store('logos', 'public');
            Setting::set('logo', $path);
        }

        return back()->with('success', 'Settings updated successfully!');
    }

    public function tickets()
    {
        $tickets = Ticket::with('user')->latest()->get();
        return view('admin.tickets', compact('tickets'));
    }

    public function showTicket(Ticket $ticket)
    {
        $ticket->load(['user', 'replies.user', 'attachments']);
        return view('admin.ticket-show', compact('ticket'));
    }

    public function replyTicket(Request $request, Ticket $ticket)
    {
        $validated = $request->validate([
            'message' => ['required', 'string'],
            'status' => ['required', 'in:open,in_progress,closed'],
            'attachments.*' => ['file', 'max:10240'],
        ]);

        $reply = TicketReply::create([
            'ticket_id' => $ticket->id,
            'user_id' => auth()->id(),
            'message' => $validated['message'],
        ]);

        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('ticket-attachments', 'public');
                
                TicketAttachment::create([
                    'ticket_reply_id' => $reply->id,
                    'file_path' => $path,
                    'file_name' => $file->getClientOriginalName(),
                    'file_type' => $file->getMimeType(),
                    'file_size' => $file->getSize(),
                ]);
            }
        }

        $ticket->update(['status' => $validated['status']]);

        // Send email notification
        Mail::to($ticket->user->email)->send(new TicketReplyMail($ticket, $reply));

        return back()->with('success', 'Reply sent successfully!');
    }

    public function knowledgeBase()
    {
        $items = KnowledgeBase::orderBy('order')->get();
        return view('admin.knowledgebase', compact('items'));
    }

    public function storeKnowledgeBase(Request $request)
    {
        $validated = $request->validate([
            'title' => ['required', 'string', 'max:255'],
            'content' => ['required', 'string'],
            'category' => ['nullable', 'string', 'max:255'],
            'order' => ['nullable', 'integer'],
        ]);

        KnowledgeBase::create($validated);

        return back()->with('success', 'Knowledge base item created successfully!');
    }

    public function deleteKnowledgeBase($id)
    {
        KnowledgeBase::findOrFail($id)->delete();
        return back()->with('success', 'Knowledge base item deleted successfully!');
    }
}

